<?php
require_once __DIR__ . '/../config.php';
if (empty($_SESSION['admin_logged_in'])) { 
    header('Location: ' . BASE_URL . '/admin/login.php'); 
    exit; 
}

// Fetch all tickets for search/filter
$pdo = getPDO();
$stmt = $pdo->query("
    SELECT t.*, e.title AS event_title 
    FROM tickets t 
    JOIN events e ON t.event_id = e.id
    ORDER BY t.purchased_at DESC
");
$tickets = $stmt->fetchAll();
?>
<!doctype html>
<html lang="en">
<head>
<meta charset="utf-8">
<title>QR Scanner & Ticket Search</title>
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<link rel="stylesheet" href="<?=BASE_URL?>/assets/css/style.css">
<style>
/* Card styling */
.card { padding:20px; margin:20px auto; max-width: 1000px; background:#f9f9f9; border-radius:10px; box-shadow:0 3px 8px rgba(0,0,0,0.15);}
h2 { text-align:center; }
/* Scanner box */
#reader { width: 100%; max-width: 420px; margin: 0 auto 20px auto; }
/* Status messages */
.status-valid { color: green; font-weight:bold; }
.status-used { color: orange; font-weight:bold; }
.alert { color:red; font-weight:bold; }
/* Search box */
.search-box { margin: 20px 0; text-align:center; }
.search-box input { padding: 8px; width: 250px; margin-right: 6px; border-radius:4px; border:1px solid #ccc; }
.search-box button { padding: 8px 12px; border:none; border-radius:4px; background:#007BFF; color:#fff; cursor:pointer; }
.search-box button:hover { background:#0056b3; }
/* Table styling */
table { width: 100%; border-collapse: collapse; margin-top: 10px; font-size:14px; }
th, td { padding: 8px; border: 1px solid #ddd; text-align:left; }
th { background: #007BFF; color: #fff; }
tr.used { background: #ffe8d6; }
tr.valid { background: #d4edda; }
img { max-width:50px; border-radius:4px; }
/* Buttons */
.btn { display:inline-block; padding:6px 12px; background:#007BFF; color:#fff; text-decoration:none; border-radius:4px; }
.btn:hover { background:#0056b3; }
.scanner-controls { text-align:center; margin:10px 0; }
</style>
</head>
<body>
<main class="container">
  <div class="card">
    <h2>QR Scanner & Ticket Management</h2>
    <p style="text-align:center;">Allow camera access to scan tickets. Or search manually by name or token.</p>

    <!-- Scanner -->
    <div id="reader"></div>
    <div class="scanner-controls">
        <button id="startScanner" class="btn">Start Scanner</button>
        <button id="stopScanner" class="btn">Stop Scanner</button>
    </div>
    <div id="scan-result" style="text-align:center; margin:10px;"></div>

    <!-- Search -->
    <div class="search-box">
        <label>Search Tickets:</label>
        <input type="text" id="ticketSearch" placeholder="Type name or token...">
    </div>

    <!-- Tickets Table -->
    <table id="ticketsTable">
        <thead>
            <tr>
                <th>ID</th>
                <th>Event</th>
                <th>Name</th>
                <th>Email</th>
                <th>Type</th>
                <th>Status</th>
                <th>Purchased At</th>
                <th>QR</th>
            </tr>
        </thead>
        <tbody>
            <?php foreach($tickets as $t): ?>
            <tr class="<?=($t['status']==='used')?'used':'valid'?>">
                <td><?=intval($t['id'])?></td>
                <td><?=htmlspecialchars($t['event_title'])?></td>
                <td><?=htmlspecialchars($t['buyer_name'])?></td>
                <td><?=htmlspecialchars($t['buyer_email'])?></td>
                <td><?=htmlspecialchars($t['ticket_type'])?></td>
                <td><?=htmlspecialchars($t['status'])?></td>
                <td><?=htmlspecialchars($t['purchased_at'])?></td>
                <td>
                    <?php if($t['qr_path'] && file_exists(__DIR__ . '/../' . $t['qr_path'])): ?>
                        <img src="<?=BASE_URL?>/<?=htmlspecialchars($t['qr_path'])?>" alt="QR">
                    <?php else: ?>
                        N/A
                    <?php endif; ?>
                </td>
            </tr>
            <?php endforeach; ?>
        </tbody>
    </table>

    <p style="text-align:center; margin-top:15px;"><a class="btn" href="<?=BASE_URL?>/admin/dashboard.php">Back to Dashboard</a></p>
  </div>
</main>

<script src="https://unpkg.com/html5-qrcode@2.3.8/minified/html5-qrcode.min.js"></script>
<script>
// DOM elements
const resultDiv = document.getElementById('scan-result');
function setMessage(html){ resultDiv.innerHTML = html; }

const ticketsTable = document.getElementById('ticketsTable').getElementsByTagName('tbody')[0];
const searchInput = document.getElementById('ticketSearch');

// Initialize scanner
const html5QrCode = new Html5Qrcode("reader");
let scannerStarted = false;

async function startScanner() {
    if(scannerStarted) return;
    try {
        const cameras = await Html5Qrcode.getCameras();
        if(cameras && cameras.length){
            const cameraId = cameras[0].id;
            await html5QrCode.start(
                cameraId,
                { fps: 10, qrbox: 250 },
                decodedText => handleScan(decodedText),
                errorMessage => {}
            );
            scannerStarted = true;
            setMessage('<div style="color:blue;">Scanner started</div>');
        } else {
            setMessage('<div class="alert">No camera found</div>');
        }
    } catch(err){
        setMessage('<div class="alert">Camera error: '+err+'</div>');
    }
}

function stopScanner() {
    if(scannerStarted){
        html5QrCode.stop().then(() => {
            scannerStarted = false;
            setMessage('<div style="color:orange;">Scanner stopped</div>');
        }).catch(err=>{
            setMessage('<div class="alert">Error stopping scanner: '+err+'</div>');
        });
    }
}

// Handle scanned QR
function handleScan(token){
    html5QrCode.pause();
    setMessage('Checking ticket...');
    fetch('<?=BASE_URL?>/api/verify.php', {
        method:'POST',
        headers:{'Content-Type':'application/json'},
        body:JSON.stringify({ token })
    }).then(r=>r.json()).then(data=>{
        if(data.status==='ok' && data.result==='valid'){
            setMessage('<div class="status-valid">Valid — ticket marked used</div>');
            highlightTicket(token,'valid');
        } else if(data.status==='ok' && data.result==='used'){
            setMessage('<div class="status-used">Already Used</div>');
            highlightTicket(token,'used');
        } else {
            setMessage('<div class="alert">Invalid ticket</div>');
        }
        setTimeout(()=> html5QrCode.resume(), 2200);
    }).catch(err=>{
        setMessage('<div class="alert">Server error</div>');
        setTimeout(()=> html5QrCode.resume(), 2500);
    });
}

// Highlight scanned ticket in table
function highlightTicket(token, status){
    Array.from(ticketsTable.getElementsByTagName('tr')).forEach(row=>{
        if(row.textContent.includes(token)){
            row.className = status==='valid' ? 'valid' : 'used';
            row.scrollIntoView({behavior:'smooth', block:'center'});
        }
    });
}

// Ticket search filter
searchInput.addEventListener('keyup', function(){
    const filter = searchInput.value.toLowerCase();
    Array.from(ticketsTable.getElementsByTagName('tr')).forEach(row=>{
        const text = row.textContent.toLowerCase();
        row.style.display = text.includes(filter) ? '' : 'none';
    });
});

// Buttons
document.getElementById('startScanner').addEventListener('click', startScanner);
document.getElementById('stopScanner').addEventListener('click', stopScanner);

// Start scanner automatically on load
startScanner();
</script>
</body>
</html>
