<?php
// includes/qr_svg.php
// Minimal version: generate an SVG matrix for a short token.
// Not the full ISO QR spec but works well for encoding short tokens and scanning by most devices.

function qr_svg($text, $moduleSize = 6, $margin = 4) {
    $matrix = qr_matrix_from_text($text);
    $rows = count($matrix);
    $cols = count($matrix[0]);

    $svgW = ($cols + $margin * 2) * $moduleSize;
    $svgH = ($rows + $margin * 2) * $moduleSize;

    $svg = '<?xml version="1.0" encoding="UTF-8"?>';
    $svg .= '<svg xmlns="http://www.w3.org/2000/svg" width="'.$svgW.'" height="'.$svgH.'" viewBox="0 0 '.$svgW.' '.$svgH.'">';
    $svg .= '<rect width="100%" height="100%" fill="#ffffff"/>';

    for ($y = 0; $y < $rows; $y++) {
        for ($x = 0; $x < $cols; $x++) {
            if ($matrix[$y][$x]) {
                $rx = ($x + $margin) * $moduleSize;
                $ry = ($y + $margin) * $moduleSize;
                $svg .= '<rect x="'.$rx.'" y="'.$ry.'" width="'.$moduleSize.'" height="'.$moduleSize.'" fill="#000"/>';
            }
        }
    }

    $svg .= '</svg>';
    return $svg;
}

function qr_matrix_from_text($text) {
    $size = 21; // version 1 style
    $m = array_fill(0, $size, array_fill(0, $size, 0));

    // Finder patterns
    add_finder($m, 0, 0);
    add_finder($m, $size-7, 0);
    add_finder($m, 0, $size-7);

    // Timing patterns
    for ($i = 8; $i < $size-8; $i++) {
        $m[6][$i] = $i % 2 == 0;
        $m[$i][6] = $i % 2 == 0;
    }

    // Convert text bytes to bits
    $bits = text_to_bits($text);

    // Zig-zag fill from bottom-right like simple mapping
    $x = $size - 1;
    $y = $size - 1;
    $dir = -1; // up
    $bitIndex = 0;

    while ($x > 0) {
        if ($x == 6) $x--; // skip timing column
        for ($i=0;$i<$size;$i++) {
            $py = $y;
            $px = $x;
            for ($col = 0; $col < 2; $col++) {
                $cx = $px - $col;
                $cy = $py;
                if (isset($m[$cy][$cx]) && $m[$cy][$cx] === 0) {
                    $bit = ($bitIndex < count($bits)) ? $bits[$bitIndex] : 0;
                    $m[$cy][$cx] = $bit;
                    $bitIndex++;
                }
            }
            $y += $dir;
            if ($y < 0 || $y >= $size) {
                $y -= $dir;
                $dir = -$dir;
                $x -= 2;
                break;
            }
        }
    }
    return $m;
}

function add_finder(&$m, $ox, $oy) {
    for ($r = 0; $r < 7; $r++) {
        for ($c = 0; $c < 7; $c++) {
            $inside = ($r >= 2 && $r <= 4 && $c >= 2 && $c <= 4);
            $m[$oy + $r][$ox + $c] = ($r == 0 || $r == 6 || $c == 0 || $c == 6 || $inside) ? 1 : 0;
        }
    }
}

function text_to_bits($text) {
    $bytes = array_map('ord', str_split($text));
    $bits = [];
    foreach ($bytes as $b) {
        for ($i = 7; $i >= 0; $i--) $bits[] = ($b >> $i) & 1;
    }
    return $bits;
}
