<?php
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}
require 'config.php'; // $mysqli

// Get search & category filters
$search = isset($_GET['q']) ? trim($_GET['q']) : '';
$category = isset($_GET['category']) ? trim($_GET['category']) : 'All';

// ----------------------------------------
// Fetch all categories
// ----------------------------------------
$categories = [];
$catRes = $mysqli->query("SELECT DISTINCT category FROM products WHERE category IS NOT NULL AND category <> '' ORDER BY category ASC");
while ($r = $catRes->fetch_assoc()) {
    $categories[] = $r['category'];
}
$catRes->free();

// ----------------------------------------
// Build product query
// ----------------------------------------
$clauses = [];
$types = '';
$params = [];

if ($search !== '') {
    $clauses[] = "(p.name LIKE ? OR p.description LIKE ? OR u.brand_name LIKE ? OR u.location LIKE ?)";
    $like = "%{$search}%";
    $types .= 'ssss';
    $params = [$like, $like, $like, $like];
}
if ($category !== '' && $category !== 'All') {
    $clauses[] = "p.category = ?";
    $types .= 's';
    $params[] = $category;
}

$where = count($clauses) ? ' WHERE ' . implode(' AND ', $clauses) : '';

$sql = "
    SELECT 
        p.id,
        p.user_id AS seller_id,
        p.name, 
        p.category, 
        p.description, 
        p.price, 
        p.image,
        u.brand_name, 
        u.location, 
        u.phone, 
        u.is_verified
    FROM products p
    LEFT JOIN users u ON p.user_id = u.id
    {$where}
    ORDER BY p.id DESC
    LIMIT 500
";

$stmt = $mysqli->prepare($sql);
if ($types !== '') {
    array_unshift($params, $types);
    $refs = [];
    foreach ($params as $key => $value) $refs[$key] = &$params[$key];
    call_user_func_array([$stmt, 'bind_param'], $refs);
}
$stmt->execute();
$result = $stmt->get_result();

// ----------------------------------------
// Handle purchase submission
// ----------------------------------------
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['buy_now'])) {
    $buyer_name = $_POST['buyer_name'] ?? '';
    $buyer_phone = $_POST['buyer_phone'] ?? '';
    $buyer_location = $_POST['buyer_location'] ?? '';
    $product_name = $_POST['product_name'] ?? '';
    $brand_name = $_POST['brand_name'] ?? '';
    $seller_phone = $_POST['seller_phone'] ?? '';

    if ($buyer_name && $buyer_phone && $buyer_location) {
        $insert = $mysqli->prepare(
            "INSERT INTO purchases (buyer_name, buyer_phone, buyer_location, product_name, brand_name, seller_phone, created_at)
            VALUES (?, ?, ?, ?, ?, ?, NOW())"
        );
        $insert->bind_param("ssssss", $buyer_name, $buyer_phone, $buyer_location, $product_name, $brand_name, $seller_phone);
        $insert->execute();
    }

    // Redirect to WhatsApp
    $message = urlencode("New Order:\nProduct: $product_name\nBuyer: $buyer_name\nPhone: $buyer_phone\nLocation: $buyer_location\nDelivery: Yes");
    header("Location: https://wa.me/$seller_phone?text=$message");
    exit;
}

// ----------------------------------------
// SEO: dynamic titles, meta, canonical
// ----------------------------------------
$page_title = $category !== 'All' ? htmlspecialchars($category) . " - ShopGRL" : "ShopGRL - Fashion, Gadgets & More";
$meta_description = $category !== 'All' ? "ShopGRL's {$category} products: Buy from verified sellers securely online." : "ShopGRL on its-insight.com: Buy premium fashion, gadgets, and more from verified sellers. Secure online shopping!";
$meta_keywords = "ShopGRL, its-insight.com, fashion, gadgets, online shopping, verified sellers, buy online";

$canonical_url = "https://its-insight.com" . strtok($_SERVER["REQUEST_URI"], '?');
?>

<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8">
<meta name="viewport" content="width=device-width, initial-scale=1">
<title><?= $page_title ?></title>
<meta name="description" content="<?= $meta_description ?>">
<meta name="keywords" content="<?= $meta_keywords ?>">
<meta name="robots" content="index, follow">
<meta name="author" content="ShopGRL / Gen(I)">
<link rel="canonical" href="<?= $canonical_url ?>">

<!-- Open Graph -->
<meta property="og:title" content="<?= $page_title ?>">
<meta property="og:description" content="<?= $meta_description ?>">
<meta property="og:type" content="website">
<meta property="og:url" content="<?= $canonical_url ?>">
<meta property="og:image" content="https://its-insight.com/assets/og-image.jpg">

<!-- JSON-LD for Store & Breadcrumb -->
<script type="application/ld+json">
{
  "@context": "https://schema.org",
  "@type": "Store",
  "name": "ShopGRL",
  "url": "https://its-insight.com/",
  "logo": "https://its-insight.com/assets/logo.png",
  "description": "ShopGRL - premium fashion, gadgets, and more from verified sellers.",
  "address": {
    "@type": "PostalAddress",
    "addressLocality": "Accra",
    "addressRegion": "Greater Accra",
    "addressCountry": "GH"
  },
  "sameAs": [
    "https://www.instagram.com/shopgrl",
    "https://www.facebook.com/shopgrl"
  ]
}
</script>

<link rel="stylesheet" href="style.css?v=<?= time() ?>">
<link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css">
</head>
<script>
if ('serviceWorker' in navigator) {
    navigator.serviceWorker.register('/sw.js')
    .then(() => console.log('Service Worker Registered'))
    .catch(err => console.error('Service Worker registration failed:', err));
}
</script>

<body>
<header class="header">
    <div class="logo">Shop<span>GRL</span></div>
    
    <form method="get" class="search-bar">
        <input type="search" name="q" placeholder="🔍 Search..." value="<?= htmlspecialchars($search) ?>">
        <?php if($category !== 'All'): ?>
            <input type="hidden" name="category" value="<?= htmlspecialchars($category) ?>">
        <?php endif; ?>
        <button type="submit">Search</button>
    </form>

    <a href="my_purchases.php" class="my-purchases-btn">🧾 My Purchases</a>
    <a href="<?= isset($_SESSION['user_id']) ? 'dashboard.php' : 'login.php' ?>" class="signin-icon">👤</a>
</header>

<section class="banner">
    <div class="banner-content">
        <h1>🔥 <?= htmlspecialchars($category==='All' ? 'Exclusive Deals!' : $category.' Products') ?></h1>
        <p>Shop premium fashion, gadgets, and more — up to <strong>50% OFF</strong> this week only.</p>
        <a href="index.php" class="banner-btn">Shop Now</a>
    </div>
</section>

<section class="categories">
    <h2>🛍️ Categories</h2>
    <div class="cat-grid">
        <a class="cat <?= ($category==='All')?'active':'' ?>" href="index.php">✨ All</a>
        <?php foreach($categories as $catName):
            $active = ($catName === $category) ? 'active' : '';
        ?>
        <a class="cat <?= $active ?>" href="index.php?category=<?= urlencode($catName) ?>">🛒 <?= htmlspecialchars($catName) ?></a>
        <?php endforeach; ?>
    </div>
</section>

<main class="products">
<h2><?= ($category !== 'All') ? '🛍️ '.htmlspecialchars($category).' Products' : '🔥 All Products' ?></h2>
<div class="grid">
<?php if($result && $result->num_rows > 0): ?>
    <?php while($p = $result->fetch_assoc()):
        $img = !empty($p['image']) ? 'uploads/'.$p['image'] : 'assets/no-image.png';
        ?>
        <article class="product-card">
            <img loading="lazy" src="<?= htmlspecialchars($img) ?>" alt="<?= htmlspecialchars($p['name']) ?>">
            <h3><?= htmlspecialchars($p['name']) ?></h3>
          <p>
            
<p class="brand">
  🏷️ <?= htmlspecialchars($p['brand_name']) ?>
  <?php if (!empty($p['is_verified']) && $p['is_verified'] == 1): ?>
    <span class="verified-badge" title="Verified Seller" onclick="showBadge()">
      <img src="grl.svg" alt="Verified"> Verified
    </span>
  <?php endif; ?>
</p>


            <p>Category: <?= htmlspecialchars($p['category']) ?></p>
            <p>Location: <?= htmlspecialchars($p['location']) ?></p>
            <p>₵<?= number_format($p['price'],2) ?>   </p>
            <p><?= htmlspecialchars($p['description']) ?></p>
            <button class="buy-btn" onclick="openPopup('<?= htmlspecialchars($p['name']) ?>','<?= htmlspecialchars($p['brand_name']) ?>','<?= htmlspecialchars($p['phone']) ?>')">💳 Buy Now</button>

            <!-- JSON-LD Product -->
            <script type="application/ld+json">
            {
                "@context": "https://schema.org",
                "@type": "Product",
                "name": "<?= htmlspecialchars($p['name']) ?>",
                "image": "<?= htmlspecialchars($img) ?>",
                "description": "<?= htmlspecialchars($p['description']) ?>",
                "brand": { "@type": "Brand", "name": "<?= htmlspecialchars($p['brand_name']) ?>" },
                "offers": {
                    "@type": "Offer",
                    "priceCurrency": "GHS",
                    "price": "<?= $p['price'] ?>",
                    "availability": "https://schema.org/InStock"
                }
            }
            </script>

        </article>
    <?php endwhile; ?>
<?php else: ?>
    <p>😞 No products found.</p>
<?php endif; ?>
</div>
</main>
<!-- Badge Popup -->
<div id="badgeModal" class="badge-modal">
  <div class="badge-modal-content">
    <span class="badge-close" onclick="closeBadge()">&times;</span>
    <img src="grl.svg" alt="Verified Badge">
    <h3>✅ Verified Seller</h3>
    <p>This seller has been officially verified by ShopGRL. You can trust their listings.</p>
  </div>
</div>
<script>
function showBadge() {
  document.getElementById('badgeModal').style.display = 'flex';
}
function closeBadge() {
  document.getElementById('badgeModal').style.display = 'none';
}

// Optional: close when clicking outside
window.onclick = function(e) {
  const modal = document.getElementById('badgeModal');
  if (e.target === modal) closeBadge();
}
</script>

<!-- Order Popup -->
<div id="orderPopup">
    <div class="popup-content">
        <h3>🛍️ Place Your Order</h3>
        <form method="post">
            <input type="hidden" name="buy_now" value="1">
            <input type="hidden" id="product_name" name="product_name">
            <input type="hidden" id="brand_name" name="brand_name">
            <input type="hidden" id="seller_phone" name="seller_phone">
            <input type="text" name="buyer_name" placeholder="Your Name" required>
            <input type="text" name="buyer_phone" placeholder="Your Phone Number" required>
            <input type="text" id="buyer_location" name="buyer_location" placeholder="Detecting your location..." required>
            <button type="submit">Send Order via WhatsApp</button>
            <button type="button" onclick="closePopup()">Close</button>
        </form>
    </div>
</div>

<script>
function openPopup(product, brand, phone){
    document.getElementById('orderPopup').style.display='flex';
    document.getElementById('product_name').value=product;
    document.getElementById('brand_name').value=brand;
    document.getElementById('seller_phone').value=phone;
}
function closePopup(){ document.getElementById('orderPopup').style.display='none'; }

// Detect location
const locationInput = document.getElementById('buyer_location');
if(navigator.geolocation){
    navigator.geolocation.getCurrentPosition(async (pos)=>{
        const {latitude, longitude} = pos.coords;
        try {
            const resp = await fetch(`https://nominatim.openstreetmap.org/reverse?format=jsonv2&lat=${latitude}&lon=${longitude}`);
            const data = await resp.json();
            let loc='';
            if(data.address.city) loc+=data.address.city;
            else if(data.address.town) loc+=data.address.town;
            else if(data.address.village) loc+=data.address.village;
            if(data.address.state) loc+=', '+data.address.state;
            locationInput.value = loc;
        } catch(err){ locationInput.placeholder="Enter your location"; }
    },()=>{ locationInput.placeholder="Enter your location"; });
}else{ locationInput.placeholder="Enter your location"; }
</script>

<footer class="footer">
  <div class="footer-container">
    <div class="footer-left">
      <p>© <?= date('Y') ?> <strong>ShopGRL</strong> — All Rights Reserved</p>
    </div>
    <div class="footer-right">
      <p>Developed by <a href="#" target="_blank">Gen(I)</a></p>
    </div>
  </div>
</footer>
<button id="installBtn" style="
    display:none;
    position:fixed;
    bottom:20px;
    right:20px;
    padding:10px 15px;
    background:#000;
    color:#d4af37;
    font-weight:bold;
    border:none;
    border-radius:8px;
    cursor:pointer;
    display:flex;
    align-items:center;
    gap:8px;
">
  <img src="shopgrl.jpeg" alt="ShopGRL Logo" style="width:24px; height:24px;">
  Install ShopGRL
</button>

<script>
let deferredPrompt;
window.addEventListener('beforeinstallprompt', (e) => {
    e.preventDefault();
    deferredPrompt = e;
    const installBtn = document.getElementById('installBtn');
    installBtn.style.display = 'flex';

    installBtn.addEventListener('click', () => {
        installBtn.style.display = 'none';
        deferredPrompt.prompt();
        deferredPrompt.userChoice.then(choiceResult => {
            console.log(choiceResult.outcome); // 'accepted' or 'dismissed'
            deferredPrompt = null;
        });
    });
});

window.addEventListener('appinstalled', () => {
    console.log('ShopGRL PWA installed!');
});
</script>

</body>
</html>
